# Whisper Dictation - First Run Installation Script
# This script downloads dependencies, detects microphone, and creates configuration

param(
    [string]$ScriptRoot = $PSScriptRoot
)

$ErrorActionPreference = "Stop"

# ============================================================
# DEPENDENCY VERSION MANIFEST
# Update these versions to trigger re-download of specific components
# ============================================================
$VERSIONS = @{
    FFmpeg = @{
        Version = "7.1"
        FolderName = "ffmpeg-7.1"
        Url = "https://github.com/BtbN/FFmpeg-Builds/releases/download/latest/ffmpeg-master-latest-win64-gpl-shared.zip"
        ZipInternalFolder = "ffmpeg-master-latest-win64-gpl-shared"  # Folder inside the zip
    }

    Model = @{
        Version = "large-v3-turbo"
        FolderName = "models-large-v3-turbo"
        FileName = "ggml-large-v3-turbo.bin"
        Url = "https://huggingface.co/ggerganov/whisper.cpp/resolve/main/ggml-large-v3-turbo.bin"
    }

    WhisperCPP = @{
        Version = "1.8.1"
        # Folder names include version and build type, determined at runtime
        Builds = @{
            CUDA124 = @{
                FolderName = "whisper-1.8.1-cublas-12.4.0-bin-x64"
                Url = "https://github.com/ggml-org/whisper.cpp/releases/download/v1.8.1/whisper-cublas-12.4.0-bin-x64.zip"
                DisplayName = "CUDA 12.4 (GPU accelerated)"
            }
            CUDA118 = @{
                FolderName = "whisper-1.8.1-cublas-11.8.0-bin-x64"
                Url = "https://github.com/ggml-org/whisper.cpp/releases/download/v1.8.1/whisper-cublas-11.8.0-bin-x64.zip"
                DisplayName = "CUDA 11.8 (GPU accelerated)"
            }
            BLAS = @{
                FolderName = "whisper-1.8.1-blas-bin-x64"
                Url = "https://github.com/ggml-org/whisper.cpp/releases/download/v1.8.1/whisper-blas-bin-x64.zip"
                DisplayName = "OpenBLAS (CPU optimized)"
            }
        }
    }
}

try {

Write-Host "=== Whisper Dictation Installation ===" -ForegroundColor Cyan
Write-Host ""

# Check for aria2c and install if missing (automatic for fast downloads)
$aria2c = Get-Command aria2c -ErrorAction SilentlyContinue
if ($aria2c) {
    Write-Host "aria2c detected - fast multi-threaded downloads enabled" -ForegroundColor Green
    Write-Host ""
} else {
    Write-Host "aria2c not found - installing for 10-20x faster downloads..." -ForegroundColor Cyan
    try {
        & winget install aria2.aria2 --silent --accept-source-agreements --accept-package-agreements
        Write-Host "  aria2c installed successfully" -ForegroundColor Green
        Write-Host "  Refreshing PATH..." -ForegroundColor Gray
        $env:Path = [System.Environment]::GetEnvironmentVariable("Path","Machine") + ";" + [System.Environment]::GetEnvironmentVariable("Path","User")
        $aria2c = Get-Command aria2c -ErrorAction SilentlyContinue
        if ($aria2c) {
            Write-Host "  aria2c is now available" -ForegroundColor Green
        }
    } catch {
        Write-Host "  Failed to install aria2c, will use slower download method" -ForegroundColor Yellow
    }
    Write-Host ""
}

# Paths
$configFile = Join-Path $ScriptRoot "config.ini"

# ============================================================
# DEPENDENCY CLEANUP - Remove outdated versions
# ============================================================
Write-Host "Checking for outdated dependencies..." -ForegroundColor Cyan

# Clean up old FFmpeg folders (anything not matching current version)
Get-ChildItem $ScriptRoot -Directory | Where-Object {
    $_.Name -like "ffmpeg-*" -and $_.Name -ne $VERSIONS.FFmpeg.FolderName
} | ForEach-Object {
    Write-Host "  Removing old FFmpeg: $($_.Name)" -ForegroundColor Yellow
    Remove-Item $_.FullName -Recurse -Force -ErrorAction SilentlyContinue
}

# Clean up old model folders (anything not matching current version)
Get-ChildItem $ScriptRoot -Directory | Where-Object {
    $_.Name -like "models-*" -and $_.Name -ne $VERSIONS.Model.FolderName
} | ForEach-Object {
    Write-Host "  Removing old model folder: $($_.Name)" -ForegroundColor Yellow
    Remove-Item $_.FullName -Recurse -Force -ErrorAction SilentlyContinue
}

# Clean up old Whisper folders (anything not matching current version pattern)
$currentWhisperVersion = $VERSIONS.WhisperCPP.Version
Get-ChildItem $ScriptRoot -Directory | Where-Object {
    $_.Name -like "whisper-*" -and $_.Name -notlike "whisper-$currentWhisperVersion-*"
} | ForEach-Object {
    Write-Host "  Removing old Whisper.cpp: $($_.Name)" -ForegroundColor Yellow
    Remove-Item $_.FullName -Recurse -Force -ErrorAction SilentlyContinue
}

# Clean up llama-cpp folder (no longer used)
$llamaCppDir = Join-Path $ScriptRoot "llama-cpp"
if (Test-Path $llamaCppDir) {
    Write-Host "  Removing unused llama-cpp folder" -ForegroundColor Yellow
    Remove-Item $llamaCppDir -Recurse -Force -ErrorAction SilentlyContinue
}

Write-Host ""

# Detect GPU capabilities
Write-Host "Detecting system capabilities..." -ForegroundColor Cyan

$hasNvidiaGPU = $false
$cudaVersion = $null

try {
    # Check for NVIDIA GPU using WMI
    $gpus = Get-CimInstance Win32_VideoController | Where-Object { $_.Name -like "*NVIDIA*" }

    if ($gpus) {
        $hasNvidiaGPU = $true
        Write-Host "  NVIDIA GPU detected: $($gpus[0].Name)" -ForegroundColor Green

        # Try to detect CUDA version via nvidia-smi
        $nvidiaSmi = Get-Command nvidia-smi -ErrorAction SilentlyContinue
        if ($nvidiaSmi) {
            $smiOutput = & nvidia-smi --query-gpu=driver_version,cuda_version --format=csv,noheader 2>$null
            if ($smiOutput -match '(\d+\.\d+)$') {
                $cudaVersion = [version]$matches[1]
                Write-Host "  CUDA Version: $cudaVersion" -ForegroundColor Green
            }
        }
    } else {
        Write-Host "  No NVIDIA GPU detected - will use CPU-optimized build" -ForegroundColor Yellow
    }
} catch {
    Write-Host "  Could not detect GPU - will use CPU-optimized build" -ForegroundColor Yellow
}

# Determine which Whisper build to use
$whisperBuild = $null
$whisperBuildType = ""

if ($hasNvidiaGPU) {
    # Use CUDA build - prefer 12.4.0 unless older CUDA detected
    if ($cudaVersion -and $cudaVersion -lt [version]"12.0") {
        $whisperBuild = $VERSIONS.WhisperCPP.Builds.CUDA118
        $whisperBuildType = "CUDA118"
    } else {
        $whisperBuild = $VERSIONS.WhisperCPP.Builds.CUDA124
        $whisperBuildType = "CUDA124"
    }
} else {
    # Use CPU build with OpenBLAS for better performance than basic build
    $whisperBuild = $VERSIONS.WhisperCPP.Builds.BLAS
    $whisperBuildType = "BLAS"
}

Write-Host "  Selected build: $($whisperBuild.DisplayName)" -ForegroundColor Cyan
Write-Host ""

# Clean up unused Whisper builds (keep only the one we need)
$VERSIONS.WhisperCPP.Builds.GetEnumerator() | ForEach-Object {
    if ($_.Key -ne $whisperBuildType) {
        $unusedDir = Join-Path $ScriptRoot $_.Value.FolderName
        if (Test-Path $unusedDir) {
            Write-Host "Removing unused Whisper build: $($_.Value.FolderName)" -ForegroundColor Yellow
            Remove-Item $unusedDir -Recurse -Force -ErrorAction SilentlyContinue
        }
    }
}

# Set paths based on selected build
$ffmpegDir = Join-Path $ScriptRoot $VERSIONS.FFmpeg.FolderName
$ffmpegExe = Join-Path $ffmpegDir "bin\ffmpeg.exe"
$whisperDir = Join-Path $ScriptRoot $whisperBuild.FolderName
$whisperExe = Join-Path $whisperDir "Release\whisper-cli.exe"
$modelDir = Join-Path $ScriptRoot $VERSIONS.Model.FolderName
$modelFile = Join-Path $modelDir $VERSIONS.Model.FileName 


# Function to get remote file size
function Get-RemoteFileSize {
    param([string]$Url)
    
    try {
        $request = [System.Net.WebRequest]::Create($Url)
        $request.Method = "HEAD"
        $response = $request.GetResponse()
        $size = $response.ContentLength
        $response.Close()
        return $size
    } catch {
        return -1
    }
}

# Function to download file with progress bar (tries aria2c first, falls back to PowerShell)
function Download-FileWithProgress {
    param(
        [string]$Url,
        [string]$OutputPath,
        [string]$Description
    )

    $fileName = Split-Path $OutputPath -Leaf

    # Check if file already exists and is complete
    if (Test-Path $OutputPath) {
        $localSize = (Get-Item $OutputPath).Length
        $remoteSize = Get-RemoteFileSize -Url $Url

        if ($remoteSize -gt 0 -and $localSize -eq $remoteSize) {
            Write-Host "  $fileName already downloaded (complete)" -ForegroundColor Green
            return
        } elseif ($localSize -gt 0) {
            Write-Host "  Found incomplete/corrupted file, re-downloading..." -ForegroundColor Yellow
            Remove-Item $OutputPath -Force
        }
    }

    # Try aria2c first for fast multi-threaded download
    $aria2c = Get-Command aria2c -ErrorAction SilentlyContinue
    if ($aria2c) {
        Write-Host "  Using aria2c for fast download..." -ForegroundColor Green
        $outputDir = Split-Path $OutputPath -Parent
        Push-Location $outputDir
        & aria2c -x 16 -s 16 -k 1M --file-allocation=none --console-log-level=warn --summary-interval=5 -o "$fileName" "$Url"
        Pop-Location
        if ($LASTEXITCODE -eq 0 -and (Test-Path $OutputPath)) {
            Write-Host "  Downloaded: $fileName" -ForegroundColor Green
            return
        } else {
            Write-Host "  aria2c failed, falling back to PowerShell..." -ForegroundColor Yellow
        }
    }

    # Fallback: PowerShell download with progress bar
    try {
        $webClient = New-Object System.Net.WebClient

        # Initialize global completion flags
        $global:downloadCompleted = $false
        $global:downloadError = $null

        # Register event handlers
        Register-ObjectEvent -InputObject $webClient -EventName DownloadProgressChanged -SourceIdentifier WebClient.DownloadProgressChanged -Action {
            $percent = $EventArgs.ProgressPercentage
            $received = [math]::Round($EventArgs.BytesReceived / 1MB, 2)
            $total = [math]::Round($EventArgs.TotalBytesToReceive / 1MB, 2)

            Write-Progress -Activity $Event.MessageData.Description `
                           -Status "$received MB of $total MB ($percent%)" `
                           -PercentComplete $percent `
                           -Id 1
        } -MessageData @{ Description = $Description } | Out-Null

        Register-ObjectEvent -InputObject $webClient -EventName DownloadFileCompleted -SourceIdentifier WebClient.DownloadFileCompleted -Action {
            $global:downloadCompleted = $true
            if ($EventArgs.Error) {
                $global:downloadError = $EventArgs.Error
            }
        } | Out-Null

        # Start async download
        $webClient.DownloadFileAsync($Url, $OutputPath)

        # Wait for download to complete - check GLOBAL variable
        while (-not $global:downloadCompleted) {
            Start-Sleep -Milliseconds 100
        }

        # Clean up events
        Unregister-Event -SourceIdentifier WebClient.DownloadProgressChanged -ErrorAction SilentlyContinue
        Unregister-Event -SourceIdentifier WebClient.DownloadFileCompleted -ErrorAction SilentlyContinue

        # IMPORTANT: Clear the progress bar completely
        Write-Progress -Activity $Description -Id 1 -Completed

        # Check for errors
        if ($global:downloadError) {
            $webClient.Dispose()
            Remove-Variable downloadCompleted -Scope Global -ErrorAction SilentlyContinue
            Remove-Variable downloadError -Scope Global -ErrorAction SilentlyContinue
            throw $global:downloadError
        }

        $webClient.Dispose()

        # Clean up global variables
        Remove-Variable downloadCompleted -Scope Global -ErrorAction SilentlyContinue
        Remove-Variable downloadError -Scope Global -ErrorAction SilentlyContinue

        Write-Host "  Downloaded: $fileName" -ForegroundColor Green

    } catch {
        Write-Host "  ERROR: Failed to download $fileName" -ForegroundColor Red
        Write-Host "  $_" -ForegroundColor Red
        Write-Host "  You can re-run this script to retry the download." -ForegroundColor Yellow

        # Clean up events and progress on error
        Unregister-Event -SourceIdentifier WebClient.DownloadProgressChanged -ErrorAction SilentlyContinue
        Unregister-Event -SourceIdentifier WebClient.DownloadFileCompleted -ErrorAction SilentlyContinue
        Write-Progress -Activity $Description -Id 1 -Completed
        Remove-Variable downloadCompleted -Scope Global -ErrorAction SilentlyContinue
        Remove-Variable downloadError -Scope Global -ErrorAction SilentlyContinue

        throw
    }
}


# Check FFmpeg - prefer system PATH version
$systemFFmpeg = Get-Command ffmpeg -ErrorAction SilentlyContinue
if ($systemFFmpeg) {
    $ffmpegExe = $systemFFmpeg.Source
    Write-Host "Using system FFmpeg from PATH: $ffmpegExe" -ForegroundColor Green
} elseif (-not (Test-Path $ffmpegExe)) {
    Write-Host "Step 1/3: Downloading FFmpeg..." -ForegroundColor Cyan

    $ffmpegZip = Join-Path $ScriptRoot "ffmpeg.zip"

    Download-FileWithProgress -Url $VERSIONS.FFmpeg.Url -OutputPath $ffmpegZip -Description "Downloading FFmpeg (~100 MB)"

    Write-Host "Extracting FFmpeg..." -ForegroundColor Yellow

    # Extract to temp location first (zip contains internal folder)
    $tempExtractPath = Join-Path $ScriptRoot "temp-ffmpeg-extract"
    if (Test-Path $tempExtractPath) {
        Remove-Item $tempExtractPath -Recurse -Force
    }
    Expand-Archive -Path $ffmpegZip -DestinationPath $tempExtractPath -Force

    # Move the internal folder to our versioned folder name
    $extractedFolder = Join-Path $tempExtractPath $VERSIONS.FFmpeg.ZipInternalFolder
    Move-Item $extractedFolder $ffmpegDir -Force

    # Clean up
    Remove-Item $tempExtractPath -Recurse -Force
    Remove-Item $ffmpegZip -Force

    Write-Host "  FFmpeg extracted successfully" -ForegroundColor Green
} else {
    Write-Host "FFmpeg already installed" -ForegroundColor Green
}

# Check and download Whisper.cpp
if (-not (Test-Path $whisperExe)) {
    Write-Host ""
    Write-Host "Step 2/3: Downloading Whisper.cpp ($($whisperBuild.DisplayName))..." -ForegroundColor Cyan

    $whisperZip = Join-Path $ScriptRoot "whisper.zip"

    # Determine download size for description
    $sizeDescription = switch ($whisperBuildType) {
        "CUDA124" { "~429 MB" }
        "CUDA118" { "~49 MB" }
        default { "~16 MB" }
    }

    Download-FileWithProgress -Url $whisperBuild.Url -OutputPath $whisperZip -Description "Downloading Whisper.cpp ($sizeDescription)"

    Write-Host "Extracting Whisper.cpp..." -ForegroundColor Yellow

    # Whisper zips extract directly to Release/ folder (no internal wrapper folder)
    # Extract directly to our versioned folder name
    if (-not (Test-Path $whisperDir)) {
        New-Item -ItemType Directory -Path $whisperDir -Force | Out-Null
    }
    Expand-Archive -Path $whisperZip -DestinationPath $whisperDir -Force

    # Clean up
    Remove-Item $whisperZip -Force

    Write-Host "  Whisper.cpp extracted successfully" -ForegroundColor Green
} else {
    Write-Host "Whisper.cpp already installed" -ForegroundColor Green
}


# Check and download AI model
if (-not (Test-Path $modelFile)) {
    Write-Host ""
    Write-Host "Step 3/3: Downloading AI model..." -ForegroundColor Cyan

    # Create models directory
    if (-not (Test-Path $modelDir)) {
        New-Item -ItemType Directory -Path $modelDir -Force | Out-Null
    }

    Download-FileWithProgress -Url $VERSIONS.Model.Url -OutputPath $modelFile -Description "Downloading AI model (~1.6 GB - this will take a while)"
    Write-Host "  AI model downloaded successfully" -ForegroundColor Green
} else {
    Write-Host "AI model already present" -ForegroundColor Green
}


# Detect microphone using ffmpeg
Write-Host ""
Write-Host "Detecting microphone..." -ForegroundColor Cyan

# Run ffmpeg and capture stderr (where device list is written)
$processInfo = New-Object System.Diagnostics.ProcessStartInfo
$processInfo.FileName = $ffmpegExe
$processInfo.Arguments = "-list_devices true -f dshow -i dummy"
$processInfo.RedirectStandardError = $true
$processInfo.UseShellExecute = $false
$processInfo.CreateNoWindow = $true

$process = New-Object System.Diagnostics.Process
$process.StartInfo = $processInfo
$process.Start() | Out-Null
$ffmpegOutput = $process.StandardError.ReadToEnd()
$process.WaitForExit()

# Parse microphone name from ffmpeg output
# Format: [dshow @ ...] "Microphone (Device Name)" (audio)
$microphoneName = ""
if ($ffmpegOutput -match '"([^"]+)"\s+\(audio\)') {
    $microphoneName = $matches[1]
    Write-Host "  Detected: $microphoneName" -ForegroundColor Green
} else {
    Write-Host "ERROR: No microphones detected" -ForegroundColor Red
    Write-Host "Please ensure a microphone is connected and enabled in Windows Sound settings" -ForegroundColor Yellow
    exit 1
}

# Language configuration
Write-Host ""
Write-Host "Language Configuration" -ForegroundColor Cyan
$topLanguages = "en, hu, de, fr, es, it, pt, ru, zh, ja, ko, ar, nl, pl, sv, tr, uk, cs, ro, fi"
Write-Host "  Common language codes: $topLanguages" -ForegroundColor Gray
Write-Host ""
$primaryInput = Read-Host "Primary language ISO code (press Enter for 'en')"
if ([string]::IsNullOrWhiteSpace($primaryInput)) { $primaryInput = "en" }
$primaryLang = $primaryInput.Trim().ToLower()

$secondaryInput = Read-Host "Secondary language ISO code (press Enter for 'hu', or type '-' for none)"
if ([string]::IsNullOrWhiteSpace($secondaryInput)) { $secondaryInput = "hu" }
$secondaryLang = $secondaryInput.Trim().ToLower()
if ($secondaryLang -eq "-") { $secondaryLang = "" }

$builtInPrompts = @{
    "en" = "Add proper punctuation including periods, commas, and question marks. Capitalize sentences correctly."
    "hu" = "Adj hozzá megfelelő írásjeleket, beleértve a pontokat, vesszőket és kérdőjeleket. Helyesen használj nagybetűket a mondatok elején."
}
$primaryPrompt   = if ($builtInPrompts.ContainsKey($primaryLang))   { $builtInPrompts[$primaryLang] }   else { "" }
$secondaryPrompt = if ($secondaryLang -ne "" -and $builtInPrompts.ContainsKey($secondaryLang)) { $builtInPrompts[$secondaryLang] } else { "" }

Write-Host "  Primary language: $primaryLang" -ForegroundColor Green
if ($secondaryLang -ne "") {
    Write-Host "  Secondary language: $secondaryLang" -ForegroundColor Green
} else {
    Write-Host "  Secondary language: (none)" -ForegroundColor Gray
}

# Create configuration file
Write-Host ""
Write-Host "Creating configuration file..." -ForegroundColor Cyan

$configContent = @"
[Paths]
WhisperCLI=$($whisperBuild.FolderName)\Release\whisper-cli.exe
ModelPath=$($VERSIONS.Model.FolderName)\$($VERSIONS.Model.FileName)
RecordingsPath=%DOCUMENTS%\Voice Typing

[Audio]
MicrophoneName=$microphoneName

[Language]
PrimaryLanguage=$primaryLang
SecondaryLanguage=$secondaryLang
PrimaryPrompt=$primaryPrompt
SecondaryPrompt=$secondaryPrompt

[Settings]
EnableLogging=false
DevMode=false
VolumeFadeDuration=200
FirstRunComplete=true
BuildType=$($whisperBuild.DisplayName)
"@

$configContent | Out-File -FilePath $configFile -Encoding UTF8 -NoNewline

Write-Host "  Configuration saved to: $configFile" -ForegroundColor Green

# Create recordings directory
$documentsPath = [Environment]::GetFolderPath("MyDocuments")
$recordingsPath = Join-Path $documentsPath "Voice Typing"
if (-not (Test-Path $recordingsPath)) {
    New-Item -ItemType Directory -Path $recordingsPath -Force | Out-Null
    Write-Host "  Created recordings directory: $recordingsPath" -ForegroundColor Green
}

Write-Host ""
Write-Host "=== Installation Complete ===" -ForegroundColor Green
Write-Host ""
Write-Host "Configuration Summary:" -ForegroundColor Cyan
Write-Host "  Build Type: $($whisperBuild.DisplayName)" -ForegroundColor White
Write-Host "  Microphone: $microphoneName" -ForegroundColor White
Write-Host "  Primary language: $primaryLang" -ForegroundColor White
if ($secondaryLang -ne "") {
    Write-Host "  Secondary language: $secondaryLang" -ForegroundColor White
}
Write-Host "  Recordings: $recordingsPath" -ForegroundColor White
Write-Host "  FFmpeg: $ffmpegExe" -ForegroundColor White
Write-Host "  Whisper: $whisperExe" -ForegroundColor White
Write-Host "  Model: $modelFile" -ForegroundColor White
Write-Host ""
Write-Host "Installed Versions:" -ForegroundColor Cyan
Write-Host "  FFmpeg: v$($VERSIONS.FFmpeg.Version)" -ForegroundColor White
Write-Host "  Whisper.cpp: v$($VERSIONS.WhisperCPP.Version)" -ForegroundColor White
Write-Host "  Model: $($VERSIONS.Model.Version)" -ForegroundColor White
Write-Host ""

if ($hasNvidiaGPU) {
    Write-Host "GPU acceleration enabled - transcription will be fast!" -ForegroundColor Green
} else {
    Write-Host "Using CPU mode - transcription will be slower but still functional." -ForegroundColor Yellow
}

Write-Host "Deterministic text processing enabled - transcriptions will be cleaned and formatted" -ForegroundColor Green

Write-Host ""
Write-Host "Next steps:" -ForegroundColor Cyan
Write-Host "  1. Run firsh-whisper-dictation.exe to start voice typing" -ForegroundColor White
Write-Host "  2. Drag and drop audio files onto the GUI for transcription with automatic text processing" -ForegroundColor White
Write-Host "  3. For grammar correction, use a web-based AI service with the transcribed text" -ForegroundColor White
Write-Host ""

} catch {
    Write-Host ""
    Write-Host "=== Installation Error ===" -ForegroundColor Red
    Write-Host $_.Exception.Message -ForegroundColor Red
    Write-Host ""
    Write-Host "Installation failed. You can re-run this script to retry." -ForegroundColor Yellow
    Write-Host ""
}

Write-Host "Press Enter to exit..." -ForegroundColor Yellow
Read-Host
